const Otp = require("../models/otp.model.js");
const User = require("../models/user.model.js");
const sendEmail = require("../utils/sendEmail.js");
const { generateOTP } = require("../utils/otpGenerator.js");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");

// Send OTP
const sendOtp = async (req, res) => {
  try {
    const { email, type } = req.body;

    // Check if email already exists when registering
    const existingUser = await User.findOne({ email });
    if (existingUser && type !== "forgot_pwd") {
      return res.status(400).json({
        success: false,
        code: "email_exists",
        message: "Email already registered",
      });
    }

    const otp = generateOTP();
    await Otp.findOneAndUpdate({ email }, { otp }, { upsert: true, new: true });

    // Improved email content to reduce spam likelihood
    const emailSubject =
      type === "forgot_pwd"
        ? "Your Password Reset Code for JobTalk"
        : "Your Registration Code for JobTalk";

    const emailText = `
Dear User,

Thank you for using JobTalk. 

Your verification code is: ${otp}

This code will expire in 5 minutes. Please enter it on the verification page to continue.

If you did not request this code, please disregard this email.

Best regards,
The JobTalk Team
jobtalk2025@gmail.com
`;

    // Send from the user's email to improve deliverability
    await sendEmail(
      email,
      emailSubject,
      emailText,
      "jobtalk2025@gmail.com" // Explicitly set the from address
    );

    res.status(200).json({
      success: true,
      message: "OTP sent successfully",
    });
  } catch (error) {
    console.error("Error sending OTP:", error);
    res.status(500).json({
      success: false,
      message: "Failed to send OTP",
    });
  }
};

// Verify OTP
const verifyEmail = async (req, res) => {
  try {
    const { email, otp } = req.body;

    // Find the OTP in database
    const otpRecord = await Otp.findOne({
      email,
      otp,
    });

    if (!otpRecord) {
      return res.status(400).json({
        success: false,
        code: "invalid_otp",
        message: "Invalid OTP",
      });
    }

    // Check if OTP is expired
    const now = new Date();
    const otpCreatedAt = new Date(otpRecord.createdAt);
    const diffInMinutes = (now - otpCreatedAt) / (1000 * 60);

    if (diffInMinutes > 5) {
      return res.status(400).json({
        success: false,
        code: "otp_expired",
        message: "OTP has expired",
      });
    }

    // Delete the used OTP
    await Otp.deleteOne({ _id: otpRecord._id });

    res.status(200).json({
      success: true,
      message: "Email verified successfully",
    });
  } catch (error) {
    console.error("Error verifying OTP:", error);
    res.status(500).json({
      success: false,
      message: "Failed to verify OTP",
    });
  }
};

// Register User
const register = async (req, res) => {
  try {
    // Check if user already exists
    const existingUser = await User.findOne({ email: req.body.email });
    if (existingUser) {
      return res.status(400).json({
        code: "email_exists",
        message: "Email already registered",
      });
    }

    // Hash password
    const salt = await bcrypt.genSalt(10);
    req.body.password = await bcrypt.hash(req.body.password, salt);

    if (req.body.role === "TPO") {
      const existingTpo = await User.findOne({
        college: req.body.college,
      });
      if (existingTpo) {
        return res.status(400).json({
          code: "email_exists",
          message: `Already a TPO registered for ${req.body.college}`,
        });
      }
    }

    // Create user
    const user = await User.create(req.body);

    // Generate token
    const token = jwt.sign(
      { id: user._id, role: user.role },
       process.env.JWT_SECRET ||
        "jobtalk2mn948ct1m9345y134ct4c31348rvy50111920qsoqmx133m",
      { expiresIn: "30d" }
    );

    res.status(201).json({
      success: true,
      token,
      user: {
        id: user._id,
        firstName: user.firstName,
        lastName: user.lastName,
        email: user.email,
        role: user.role,
      },
    });
  } catch (error) {
    console.error("Error registering user:", error);
    res.status(500).json({
      success: false,
      message: "Failed to register user",
    });
  }
};

// Login User
const login = async (req, res) => {
  try {
    const { email, password } = req.body;

    // Find user
    const user = await User.findOne({ email });
    if (!user) {
      return res.status(404).json({
        success: false,
        code: "user_not_found",
        message: "User not found",
      });
    }

    // Check password
    const isPasswordValid = await bcrypt.compare(password, user.password);
    if (!isPasswordValid) {
      return res.status(400).json({
        success: false,
        code: "bad_pwd",
        message: "Invalid password",
      });
    }

    // Generate JWT token
    const token = jwt.sign(
      { id: user._id, role: user.role },
      process.env.JWT_SECRET || "jobtalk2mn948ct1m9345y134ct4c31348rvy50111920qsoqmx133m",
      { expiresIn: "30d" }
    );

    res.status(200).json({
      success: true,
      token,
      user: {
        id: user._id,
        firstName: user.firstName,
        lastName: user.lastName,
        email: user.email,
        role: user.role,
      },
    });
  } catch (error) {
    console.error("Error logging in:", error);
    res.status(500).json({
      success: false,
      message: "Failed to login",
    });
  }
};

// Reset Password
const resetPassword = async (req, res) => {
  try {
    const { email, new_password } = req.body;

    // Find user
    const user = await User.findOne({ email });
    if (!user) {
      return res.status(404).json({
        success: false,
        code: "user_not_found",
        message: "User not found",
      });
    }

    // Hash new password
    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(new_password, salt);

    // Update user's password
    user.password = hashedPassword;
    await user.save();

    // Send confirmation email
    const emailText = `
Hello,

Your password has been successfully reset. You can now login with your new password.

If you didn't request this change, please contact us immediately.

Thanks,  
JobSearcher Team
`;

    await sendEmail(email, "Password Reset Successful", emailText);

    res.status(200).json({
      success: true,
      message: "Password reset successful",
    });
  } catch (error) {
    console.error("Error resetting password:", error);
    res.status(500).json({
      success: false,
      message: "Failed to reset password",
    });
  }
};

module.exports = {
  sendOtp,
  verifyEmail,
  register,
  login,
  resetPassword,
};
