const User = require("../models/user.model");
const Job = require("../models/job.model");
const Application = require("../models/application.model");

exports.getEmployerJobs = async (req, res) => {
  try {
    const employerJobs = await Job.find({ createdBy: req.user.id });

    if (!employerJobs || employerJobs.length === 0) {
      return res.status(404).json({ message: "No jobs found" });
    }

    res.status(200).json({ employerJobs });
  } catch (error) {
    console.error("Error fetching employer jobs:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};

exports.getReceivedApplications = async (req, res) => {
  try {
    const { jobId, status, dateRange, page = 1, limit = 6 } = req.query;

    // Validate and sanitize input parameters
    const sanitizedPage = Math.max(1, parseInt(page));
    const sanitizedLimit = Math.min(50, Math.max(1, parseInt(limit)));

    // Find the job by ID
    const job = await Job.findOne({
      _id: jobId,
      createdBy: req.user.id, // Ensure the job belongs to the employer
    });

    if (!job) {
      return res.status(404).json({ message: "Job not found" });
    }

    // Build query for applications
    let query = { job: job._id };

    // Add status filter if provided
    if (status && status !== "all") {
      const validStatuses = ["pending", "shortlisted", "accepted", "rejected"];
      if (validStatuses.includes(status.toLowerCase())) {
        query.status = status.toLowerCase();
      }
    }

    // Add date range filter if provided
    if (dateRange && dateRange !== "all") {
      const now = new Date();
      const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());

      switch (dateRange) {
        case "today":
          query.createdAt = { $gte: today };
          break;
        case "week":
          const weekAgo = new Date(today);
          weekAgo.setDate(weekAgo.getDate() - 7);
          query.createdAt = { $gte: weekAgo };
          break;
        case "month":
          const monthAgo = new Date(today);
          monthAgo.setMonth(monthAgo.getMonth() - 1);
          query.createdAt = { $gte: monthAgo };
          break;
        default:
          // Invalid dateRange, ignore it
          break;
      }
    }

    // Calculate pagination
    const skip = (sanitizedPage - 1) * sanitizedLimit;

    // Fetch applications with populated user data
    const [applications, total] = await Promise.all([
      Application.find(query)
        .populate({
          path: "user",
          select: "firstName lastName email phone skills experience",
        })
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(sanitizedLimit),
      Application.countDocuments(query),
    ]);

    // Calculate total pages
    const totalPages = Math.ceil(total / sanitizedLimit);

    // Ensure current page is valid
    const currentPage = Math.min(sanitizedPage, totalPages || 1);

    res.status(200).json({
      applications,
      pagination: {
        currentPage,
        totalPages,
        totalItems: total,
        itemsPerPage: sanitizedLimit,
        hasNextPage: currentPage < totalPages,
        hasPreviousPage: currentPage > 1,
      },
    });
  } catch (error) {
    console.error("Error fetching received applications:", error);
    res.status(500).json({ message: error.message });
  }
};

exports.getUserDetails = async (req, res) => {
  try {
    const user = await User.findById(req.params.id).select(
      "-password -savedJobs -appliedJobs -companyDescription -companyLogo -companyName"
    );
    res.status(200).json({ user });
  } catch (error) {
    console.error("Error fetching user details:", error);
    res.status(500).json({ message: error.message });
  }
};

exports.updateStatus = async (req, res) => {
  try {
    const { applicationId, status } = req.body;

    const application = await Application.findByIdAndUpdate(applicationId, {
      status,
    });

    if (!application) {
      return res.status(404).json({ message: "Application not found" });
    }

    res
      .status(200)
      .json({ message: "Status updated successfully", application });
  } catch (error) {
    console.error("Error updating status:", error);
    res.status(500).json({ message: error.message });
  }
};
